/*
  Copyright 2019 Kakao Corp.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 */
@file:JvmName("AuthCodeClientKt")

package com.kakao.sdk.auth

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.os.Handler
import android.os.Looper
import android.os.ResultReceiver
import com.kakao.sdk.auth.model.AuthType
import com.kakao.sdk.common.util.KakaoJson
import com.kakao.sdk.common.KakaoSdk
import com.kakao.sdk.common.util.SdkLog
import com.kakao.sdk.common.model.*
import com.kakao.sdk.common.util.IntentResolveClient
import io.reactivex.Single
import io.reactivex.SingleEmitter
import io.reactivex.SingleTransformer
import java.net.HttpURLConnection

/**
 * OAuth 2.0 Spec 의 인증 코드를 발급받기 위해 사용되는 클라이언트. (for ReactiveX)
 */
class RxAuthCodeClient(
    private val intentResolveClient: IntentResolveClient = IntentResolveClient.instance,
    private val applicationInfo: ApplicationInfo = KakaoSdk.applicationContextInfo,
    private val contextInfo: ContextInfo = KakaoSdk.applicationContextInfo
) {

    private val CAPRI_LOGGED_IN_ACTIVITY = "com.kakao.talk.intent.action.CAPRI_LOGGED_IN_ACTIVITY"

    /**
     * 카카오톡에 연결된 카카오계정으로 사용자를 인증하고 인증 코드 발급.
     *
     * @param context 카카오톡 로그인 Activity를 실행하기 위한 현재 Activity context
     *
     * @return 인증 코드를 방출하는 [Single] 반환. 발급 받은 인증 코드는 [AuthApiClient.issueAccessToken] 파라미터로 사용.
     */
    @JvmOverloads
    fun authorizeWithKakaoTalk(
        context: Context,
        requestCode: Int = AuthCodeClient.DEFAULT_REQUEST_CODE,
        channelPublicIds: List<String>? = null,
        serviceTerms: List<String>? = null
    ): Single<String> =
        Single.create<String> { emitter ->
            val clientId = applicationInfo.appKey
            val redirectUri = "kakao$clientId://oauth"
            val kaHeader = contextInfo.kaHeader
            val extras = Bundle().apply {
                channelPublicIds?.let { putString(Constants.CHANNEL_PUBLIC_ID, channelPublicIds.joinToString(",")) }
                serviceTerms?.let { putString(Constants.SERVICE_TERMS, serviceTerms.joinToString(",")) }
            }
            val talkIntent = talkLoginIntent(clientId, redirectUri, kaHeader, extras)
            val resolvedIntent = intentResolveClient.resolveTalkIntent(context, talkIntent)
            if (resolvedIntent == null) {
                emitter.onError(
                    ClientError(
                        ClientErrorCause.NotSupported,
                        "KakaoTalk not installed"
                    )
                )
                return@create
            }
            try {
                context.startActivity(
                    Intent(context, TalkAuthCodeActivity::class.java)
                        .putExtra(Constants.KEY_LOGIN_INTENT, resolvedIntent)
                        .putExtra(Constants.KEY_REQUEST_CODE, requestCode)
                        .putExtra(Constants.KEY_RESULT_RECEIVER, resultReceiver(emitter))
                )
            } catch (startActivityError: Throwable) {
                emitter.onError(startActivityError)
            }
        }.compose(handleAuthCodeError())

    /**
     * 기본 웹 브라우저(CustomTabs)에 있는 카카오계정 cookie 로 인증하여 인증 코드 발급.
     *
     * @param context CustomTabs를 실행하기 위한 현재 Activity context
     * @param authType 인증 동작 지정. [AuthType]
     * @param scopes 추가로 동의 받고자 하는 동의 항목 ID 목록. 카카오 디벨로퍼스 동의 항목 설정 화면에서 확인 가능.
     *
     * @return 인증 코드를 방출하는 [Single] 반환. 발급 받은 인증 코드는 [AuthApiClient.issueAccessToken] 파라미터로 사용.
     */
    @JvmOverloads
    fun authorizeWithKakaoAccount(
        context: Context,
        authType: AuthType? = null,
        scopes: List<String>? = null,
        agt: String? = null,
        channelPublicIds: List<String>? = null,
        serviceTerms: List<String>? = null
    ): Single<String> =
        Single.create<String> { emitter ->
            val clientId = applicationInfo.appKey
            val redirectUri = "kakao$clientId://oauth"
            val kaHeader = contextInfo.kaHeader
            val fullUri = UriUtility().authorizeUri(clientId, agt, redirectUri, scopes, kaHeader, channelPublicIds, serviceTerms, authType)
            try {
                context.startActivity(authCodeIntent(context, fullUri, redirectUri, resultReceiver(emitter)))
            } catch (startActivityError: Throwable) {
                emitter.onError(startActivityError)
            }
        }.compose(handleAuthCodeError())

    private fun baseTalkLoginIntent(): Intent =
        Intent().setAction(CAPRI_LOGGED_IN_ACTIVITY).addCategory(Intent.CATEGORY_DEFAULT)

    @JvmSynthetic
    internal fun talkLoginIntent(
        clientId: String,
        redirectUri: String,
        kaHeader: String,
        extras: Bundle
    ): Intent =
        baseTalkLoginIntent()
            .putExtra(Constants.EXTRA_APPLICATION_KEY, clientId)
            .putExtra(Constants.EXTRA_REDIRECT_URI, redirectUri)
            .putExtra(Constants.EXTRA_KA_HEADER, kaHeader)
            .putExtra(Constants.EXTRA_EXTRAPARAMS, extras)

    @JvmSynthetic
    internal fun authCodeIntent(
        context: Context,
        fullUri: Uri,
        redirectUri: String,
        resultReceiver: ResultReceiver
    ) =
        Intent(context, AuthCodeHandlerActivity::class.java)
            .putExtra(Constants.KEY_BUNDLE, Bundle().apply {
                putParcelable(Constants.KEY_RESULT_RECEIVER, resultReceiver)
                putParcelable(Constants.KEY_FULL_URI, fullUri)
                putString(Constants.KEY_REDIRECT_URI, redirectUri)
            })

    @JvmSynthetic
    internal fun resultReceiver(emitter: SingleEmitter<String>): ResultReceiver =
        object : ResultReceiver(Handler(Looper.getMainLooper())) {
            override fun onReceiveResult(resultCode: Int, resultData: Bundle?) {
                this@RxAuthCodeClient.onReceivedResult(resultCode, resultData, emitter)
            }
        }

    @JvmSynthetic
    internal fun onReceivedResult(
        resultCode: Int,
        resultData: Bundle?,
        emitter: SingleEmitter<String>
    ) {
        when (resultCode) {
            Activity.RESULT_OK -> {
                val uri = resultData?.getParcelable<Uri>(Constants.KEY_URL)
                val code = uri?.getQueryParameter(Constants.CODE)
                if (code != null) {
                    emitter.onSuccess(code)
                    return
                }
                val error = uri?.getQueryParameter(Constants.ERROR) as String
                val errorDescription = uri.getQueryParameter(Constants.ERROR_DESCRIPTION) as String
                emitter.onError(
                    AuthError(
                        HttpURLConnection.HTTP_MOVED_TEMP,
                        kotlin.runCatching {
                            KakaoJson.fromJson<AuthErrorCause>(error, AuthErrorCause::class.java)
                        }.getOrDefault(
                            AuthErrorCause.Unknown
                        ),
                        AuthErrorResponse(error, errorDescription)
                    )
                )
            }
            Activity.RESULT_CANCELED -> {
                val exception =
                    resultData?.getSerializable(Constants.KEY_EXCEPTION)
                            as KakaoSdkError
                emitter.onError(exception)
            }
            else -> throw IllegalArgumentException("Unknown resultCode in RxAuthCodeClient#onReceivedResult()")
        }
    }

    @JvmSynthetic
    internal fun <T> handleAuthCodeError(): SingleTransformer<T, T> = SingleTransformer {
        it.doOnError { SdkLog.e(it) }
            .doOnSuccess { SdkLog.i(it!!) }
    }

    companion object {
        @JvmStatic
        val instance by lazy { AuthCodeClient.rx }
    }
}

/**
 * ReactiveX 를 위한 [AuthCodeClient] singleton 객체
 */
val AuthCodeClient.Companion.rx by lazy { RxAuthCodeClient() }