/*
  Copyright 2020 Kakao Corp.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 */
@file:JvmName("LoginClientKt")

package com.kakao.sdk.auth

import android.content.Context
import com.kakao.sdk.auth.model.AuthType
import com.kakao.sdk.auth.model.OAuthToken
import io.reactivex.Single
import io.reactivex.schedulers.Schedulers

/**
 * 카카오 로그인을 실행하기 위한 클라이언트. (for ReactiveX)
 * 카카오톡 또는 웹브라우저에 있는 카카오계정 cookie를 이용해 사용자 정보를 획득하고 API 요청에 필요한 토큰을 발급 받습니다.
 */
class RxLoginClient(
    private val authApiClient: RxAuthApiClient = AuthApiClient.rx,
    private val authCodeClient: RxAuthCodeClient = AuthCodeClient.rx
) {

    /**
     * 카카오톡으로 로그인. 카카오톡에 연결된 카카오계정으로 사용자를 인증하고 [OAuthToken] 발급.
     *
     * 발급된 토큰은 [TokenManagerProvider]에 지정된 토큰 저장소에 자동으로 저장됨.
     *
     * @param context 카카오톡 로그인 Activity를 실행하기 위한 현재 Activity context
     *
     * @return [OAuthToken]을 방출하는 [Single] 반환.
     */
    @JvmOverloads
    fun loginWithKakaoTalk(
        context: Context,
        requestCode: Int = AuthCodeClient.DEFAULT_REQUEST_CODE,
        channelPublicIds: List<String>? = null,
        serviceTerms: List<String>? = null
    ) : Single<OAuthToken> =
        authCodeClient.authorizeWithKakaoTalk(context, requestCode, channelPublicIds, serviceTerms)
            .observeOn(Schedulers.io())
            .flatMap { authApiClient.issueAccessToken(it) }

    /**
     * 카카오계정으로 로그인. 기본 웹 브라우저(CustomTabs)에 있는 카카오계정 cookie 로 사용자를 인증하고 [OAuthToken] 발급.
     *
     * 발급된 토큰은 [TokenManagerProvider]에 지정된 토큰 저장소에 자동으로 저장됨.
     *
     * @param context CustomTabs를 실행하기 위한 현재 Activity context
     * @param authType 인증 동작 지정. [AuthType]
     *
     * @return [OAuthToken]을 방출하는 [Single] 반환.
     */
    @JvmOverloads
    fun loginWithKakaoAccount(
        context: Context,
        authType: AuthType? = null,
        channelPublicIds: List<String>? = null,
        serviceTerms: List<String>? = null
    ) : Single<OAuthToken> =
        authCodeClient.authorizeWithKakaoAccount(
            context,
            authType = authType,
            channelPublicIds = channelPublicIds,
            serviceTerms = serviceTerms
        )
            .observeOn(Schedulers.io())
            .flatMap { authApiClient.issueAccessToken(it) }

    /**
     * 사용자가 아직 동의하지 않은 개인정보 및 접근권한 동의 항목에 대하여 동의를 요청하는 동의 화면을 출력하고, 사용자 동의 시 동의항목이 업데이트 된 [OAuthToken] 발급.
     *
     * 발급된 토큰은 [TokenManagerProvider]에 지정된 토큰 저장소에 자동으로 저장됨.
     *
     * @param context CustomTabs를 실행하기 위한 현재 Activity context
     * @param scopes 추가로 동의 받고자 하는 동의 항목 ID 목록. 카카오 디벨로퍼스 동의 항목 설정 화면에서 확인 가능.
     *
     * @return [OAuthToken]을 방출하는 [Single] 반환.
     */
    fun loginWithNewScopes(
        context: Context,
        scopes: List<String>
    ) : Single<OAuthToken> =
        authApiClient.agt()
            .subscribeOn(Schedulers.io())
            .flatMap { authCodeClient.authorizeWithKakaoAccount(context, scopes = scopes, agt = it) }
            .observeOn(Schedulers.io())
            .flatMap { authApiClient.issueAccessToken(it) }

    companion object {
        @JvmStatic
        val instance by lazy { LoginClient.rx }
    }
}

/**
 * ReactiveX 를 위한 [LoginClient] singleton 객체
 */
val LoginClient.Companion.rx by lazy { RxLoginClient() }
