/*
  Copyright 2019 Kakao Corp.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 */
@file:JvmName("AuthApiClientKt")

package com.kakao.sdk.auth

import com.kakao.sdk.auth.model.OAuthToken
import com.kakao.sdk.auth.model.AgtResponse
import com.kakao.sdk.auth.network.rxKauth
import com.kakao.sdk.common.KakaoSdk
import com.kakao.sdk.common.util.SdkLog
import com.kakao.sdk.common.model.ApplicationInfo
import com.kakao.sdk.common.model.ClientError
import com.kakao.sdk.common.model.ClientErrorCause
import com.kakao.sdk.common.model.ContextInfo
import com.kakao.sdk.common.network.ApiFactory

import io.reactivex.Single
import io.reactivex.SingleTransformer

/**
 * 카카오 OAuth 서버에서 제공하는 API를 사용하기 위한 클라이언트. (for ReactiveX)
 */
class RxAuthApiClient(
    private val authApi: RxAuthApi = ApiFactory.rxKauth.create(RxAuthApi::class.java),
    private val tokenManagerProvider: TokenManagerProvider = TokenManagerProvider.instance,
    private val applicationInfo: ApplicationInfo = KakaoSdk.applicationContextInfo,
    private val contextInfo: ContextInfo = KakaoSdk.applicationContextInfo
) {

    /**
     * [AuthCodeClient] 를 이용하여 발급 받은 인증 코드를 사용하여 액세스 토큰을 발급.
     *
     * 발급된 토큰은 [TokenManagerProvider]에 지정된 토큰 저장소에 자동으로 저장됨.
     *
     * @param authCode 인증 코드
     *
     * @return [OAuthToken]을 방출하는 [Single] 반환.
     */
    fun issueAccessToken(
        authCode: String
    ): Single<OAuthToken> {
        val clientId = applicationInfo.appKey
        val redirectUri = "kakao$clientId://oauth"
        val androidKeyHash = contextInfo.signingKeyHash
        return authApi.issueAccessToken(
            clientId = clientId,
            redirectUri = redirectUri,
            androidKeyHash = androidKeyHash,
            authCode = authCode
        )
            .compose(handleAuthError())
            .map { OAuthToken.fromResponse(it) }
            .doOnSuccess { tokenManagerProvider.manager.setToken(it) }

    }


    /**
     * 리프레시 토큰으로 액세스 토큰을 갱신.
     *
     * 갱신된 토큰은 [TokenManagerProvider]에 지정된 토큰 저장소에 자동으로 저장됨.
     *
     * @param oldToken 이전에 발급 받았던 토큰. 기본 값은 [TokenManagerProvider]에 지정된 저장소의 토큰.
     *
     * @return [OAuthToken]을 방출하는 [Single] 반환.
     *
     * @throws ClientErrorCause.TokenNotFound
     */
    @JvmOverloads
    fun refreshAccessToken(
        oldToken: OAuthToken = tokenManagerProvider.manager.getToken() ?: throw ClientError(ClientErrorCause.TokenNotFound)
    ): Single<OAuthToken> {
        val clientId = applicationInfo.appKey
        val androidKeyHash = contextInfo.signingKeyHash
        return authApi.issueAccessToken(
            clientId = clientId,
            redirectUri = null,
            androidKeyHash = androidKeyHash,
            refreshToken = oldToken.refreshToken,
            grantType = Constants.REFRESH_TOKEN
        )
            .compose(handleAuthError())
            .map { OAuthToken.fromResponse(it, oldToken) }
            .doOnSuccess { tokenManagerProvider.manager.setToken(it) }
    }


    /**
     * @suppress
     */
    fun agt(clientId: String = KakaoSdk.applicationContextInfo.appKey): Single<String> =
        Single.just(tokenManagerProvider.manager.getToken()?.accessToken)
            .flatMap { authApi.agt(clientId, it) }
            .compose(handleAuthError())
            .map { it.agt }

    @JvmSynthetic
    internal fun <T> handleAuthError(): SingleTransformer<T, T> = SingleTransformer {
        it.onErrorResumeNext { Single.error(AuthApiClient.translateError(it)) }
            .doOnError { SdkLog.e(it) }
            .doOnSuccess { SdkLog.i(it!!) }
    }

    companion object {
        @JvmStatic
        val instance by lazy { AuthApiClient.rx }
    }
}

/**
 * ReactiveX 를 위한 [AuthApiClient] singleton 객체
 */
val AuthApiClient.Companion.rx by lazy { RxAuthApiClient() }