/*
  Copyright 2019 Kakao Corp.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 */
@file:JvmName("UserApiClientKt")

package com.kakao.sdk.user

import android.content.Context
import com.kakao.sdk.auth.AppsHandlerActivity
import com.kakao.sdk.auth.AuthApiClient
import com.kakao.sdk.auth.AuthCodeClient
import com.kakao.sdk.auth.IntentFactory
import com.kakao.sdk.auth.RxResultReceiver
import com.kakao.sdk.auth.TokenManagerProvider
import com.kakao.sdk.auth.model.OAuthToken
import com.kakao.sdk.auth.model.Prompt
import com.kakao.sdk.auth.network.RxAuthOperations
import com.kakao.sdk.auth.network.rxKapiWithOAuth
import com.kakao.sdk.auth.rx
import com.kakao.sdk.common.model.AppsError
import com.kakao.sdk.common.util.KakaoJson
import com.kakao.sdk.network.ApiFactory
import com.kakao.sdk.network.RxOperations
import com.kakao.sdk.user.model.AccessTokenInfo
import com.kakao.sdk.user.model.ScopeInfo
import com.kakao.sdk.user.model.User
import com.kakao.sdk.user.model.UserRevokedServiceTerms
import com.kakao.sdk.user.model.UserServiceTerms
import com.kakao.sdk.user.model.UserShippingAddresses
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import java.net.HttpURLConnection
import java.util.Date

/**
 * 카카오 로그인 API 클라이언트<br>Client for the Kakao Login APIs
 */
class RxUserApiClient(
    private val userApi: RxUserApi = ApiFactory.rxKapiWithOAuth.create(RxUserApi::class.java),
    val authOperations: RxAuthOperations = RxAuthOperations.instance,
    private val tokenManagerProvider: TokenManagerProvider = TokenManagerProvider.instance,
) {

    /**
     * 카카오톡으로 로그인<br>Login with Kakao Talk
     *
     * @param context Android Context
     * @param nonce ID 토큰 재생 공격 방지를 위한 검증 값, 임의의 문자열<br>Random strings to prevent replay attacks
     * @param channelPublicIds 카카오톡 채널 프로필 ID<br>Kakao Talk Channel's profile IDs
     * @param serviceTerms 서비스 약관 목록<br>List of service terms
     */
    @JvmOverloads
    fun loginWithKakaoTalk(
        context: Context,
        requestCode: Int = AuthCodeClient.DEFAULT_REQUEST_CODE,
        nonce: String? = null,
        channelPublicIds: List<String>? = null,
        serviceTerms: List<String>? = null,
    ): Single<OAuthToken> {
        val codeVerifier = AuthCodeClient.codeVerifier()
        return AuthCodeClient.rx.authorizeWithKakaoTalk(
            context,
            prompts = null,
            requestCode = requestCode,
            nonce = nonce,
            channelPublicIds = channelPublicIds,
            serviceTerms = serviceTerms,
            codeVerifier = codeVerifier
        )
            .observeOn(Schedulers.io())
            .flatMap { AuthApiClient.rx.issueAccessToken(it, codeVerifier) }
    }

    /**
     * 카카오계정으로 로그인<br>Login with Kakao Account
     *
     * @param context Android Context
     * @param prompts 동의 화면에 상호작용 추가 요청 프롬프트<br>Prompt to add an interaction to the consent screen
     * @param loginHint 카카오계정 로그인 페이지 ID에 자동 입력할 이메일 또는 전화번호, +82 00-0000-0000 형식<br>Email or phone number in the format +82 00-0000-0000 to fill in ID field of the Kakao Account login page
     * @param nonce ID 토큰 재생 공격 방지를 위한 검증 값, 임의의 문자열<br>Random strings to prevent replay attacks
     * @param channelPublicIds 카카오톡 채널 프로필 ID<br>Kakao Talk Channel's profile IDs
     * @param serviceTerms 서비스 약관 목록<br>List of service terms
     */
    @JvmOverloads
    fun loginWithKakaoAccount(
        context: Context,
        prompts: List<Prompt>? = null,
        loginHint: String? = null,
        nonce: String? = null,
        channelPublicIds: List<String>? = null,
        serviceTerms: List<String>? = null,
    ): Single<OAuthToken> {
        val codeVerifier = AuthCodeClient.codeVerifier()
        return AuthCodeClient.rx.authorizeWithKakaoAccount(
            context,
            prompts = prompts,
            loginHint = loginHint,
            nonce = nonce,
            channelPublicIds = channelPublicIds,
            serviceTerms = serviceTerms,
            codeVerifier = codeVerifier
        )
            .observeOn(Schedulers.io())
            .flatMap { AuthApiClient.rx.issueAccessToken(it, codeVerifier) }
    }

    /**
     * 추가 항목 동의 받기<br>Request additional consent
     *
     * @param context Android Context
     * @param scopes 동의 항목 ID 목록<br>List of the scope IDs
     * @param nonce ID 토큰 재생 공격 방지를 위한 검증 값, 임의의 문자열<br>A random string to prevent replay attacks
     */
    fun loginWithNewScopes(
        context: Context,
        scopes: List<String>,
        nonce: String? = null,
    ): Single<OAuthToken> {
        val codeVerifier = AuthCodeClient.codeVerifier()
        return AuthApiClient.rx.agt()
            .subscribeOn(Schedulers.io())
            .flatMap {
                AuthCodeClient.rx.authorizeWithKakaoAccount(
                    context,
                    scopes = scopes,
                    agt = it,
                    nonce = nonce,
                    codeVerifier = codeVerifier
                )
            }
            .observeOn(Schedulers.io())
            .flatMap { AuthApiClient.rx.issueAccessToken(it, codeVerifier) }
    }

    /**
     * 사용자 정보 가져오기<br>Retrieve user information
     *
     * @param properties 사용자 프로퍼티<br>User properties
     * @param secureResource 이미지 URL을 HTTPS로 설정<br>Whether to use HTTPS for the image URL
     */
    @JvmOverloads
    fun me(
        properties: List<String>? = null,
        secureReSource: Boolean = true,
    ): Single<User> = userApi.me(
        secureReSource,
        properties?.let { KakaoJson.toJson(it) },
    )
        .compose(RxOperations.handleApiError())
        .compose(authOperations.handleApiError())
        .map { it.toUser() }

    /**
     * 연결하기<br>Manual signup
     *
     * @param properties 사용자 프로퍼티<br>User properties
     */
    @JvmOverloads
    fun signup(properties: Map<String, String>? = null): Completable = userApi.signup(properties)
        .compose(RxOperations.handleCompletableError())
        .compose(authOperations.handleCompletableError())

    /**
     * 액세스 토큰 정보<br>Access token information
     */
    fun accessTokenInfo(): Single<AccessTokenInfo> = userApi.accessTokenInfo()
        .compose(RxOperations.handleApiError())
        .compose(authOperations.handleApiError())

    /**
     * 사용자 정보 저장하기<br>Store user information
     *
     * @param properties 사용자 프로퍼티<br>User properties
     */
    fun updateProfile(properties: Map<String, String>): Completable =
        userApi.updateProfile(properties)
            .compose(RxOperations.handleCompletableError())
            .compose(authOperations.handleCompletableError())

    /**
     * 로그아웃<br>Logout
     */
    fun logout(): Completable = userApi.logout()
        .compose(RxOperations.handleCompletableError())
        .compose(authOperations.handleCompletableError())
        .doOnEvent { tokenManagerProvider.manager.clear() }

    /**
     * 연결 끊기<br>Unlink
     */
    fun unlink(): Completable = userApi.unlink()
        .compose(RxOperations.handleCompletableError())
        .compose(authOperations.handleCompletableError())
        .doOnComplete { tokenManagerProvider.manager.clear() }

    /**
     * 배송지 선택하기<br>Select shipping address
     */
    fun selectShippingAddresses(context: Context): Single<Long> {
        return appsShippingAddresses(context, Constants.SELECT_SHIPPING_ADDRESS_PATH)
    }

    /**
     * 배송지 가져오기<br>Retrieve shipping address
     *
     * @param fromUpdatedAt 이전 페이지의 마지막 배송지 수정 시각, `0` 전달 시 처음부터 조회<br>Last shipping address modification on previous page, retrieve from beginning if passing `0'
     * @param pageSize 한 페이지에 포함할 배송지 수(기본값: 10)<br>Number of shipping addresses displayed on a page (Default: 10)
     */
    @JvmOverloads
    fun shippingAddresses(
        fromUpdatedAt: Date? = null,
        pageSize: Int? = null,
    ): Single<UserShippingAddresses> {
        return userApi.shippingAddresses(fromUpdatedAt = fromUpdatedAt, pageSize = pageSize)
            .compose(RxOperations.handleApiError())
            .compose(authOperations.handleApiError())
    }

    /**
     * 배송지 가져오기<br>Retrieve shipping address
     *
     * @param addressId 배송지 ID<br>Shipping address ID
     */
    fun shippingAddresses(addressId: Long): Single<UserShippingAddresses> {
        return userApi.shippingAddresses(addressId)
            .compose(RxOperations.handleApiError())
            .compose(authOperations.handleApiError())
    }

    /**
     * 동의 내역 확인하기<br>Retrieve consent details
     *
     * @param scopes 동의 항목 ID 목록<br>List of the scope IDs
     */
    fun scopes(scopes: List<String>? = null): Single<ScopeInfo> {
        return userApi.scopes(if (scopes == null) null else KakaoJson.toJson(scopes))
            .compose(RxOperations.handleApiError())
            .compose(authOperations.handleApiError())
    }

    /**
     * 동의 철회하기<br>Revoke consent
     *
     * @param scopes 동의 항목 ID 목록<br>List of the scope IDs
     */
    fun revokeScopes(scopes: List<String>): Single<ScopeInfo> {
        return userApi.revokeScopes(KakaoJson.toJson(scopes))
            .compose(RxOperations.handleApiError())
            .compose(authOperations.handleApiError())
    }

    /**
     * 서비스 약관 동의 내역 확인하기<br>Retrieve consent details for service terms
     *
     * @param tags 서비스 약관 태그 목록<br>Tags of service terms
     * @param result 조회 대상(`agreed_service_terms`: 사용자가 동의한 서비스 약관 목록 | `app_service_terms`: 앱에 사용 설정된 서비스 약관 목록, 기본값: `agreed_service_terms`)<br>Result type (`agreed_service_terms`: List of service terms the user has agreed to | `app_service_terms`: List of service terms enabled for the app, Default: `agreed_service_terms`)
     */
    fun serviceTerms(
        tags: List<String>? = null,
        result: String? = null,
    ): Single<UserServiceTerms> {
        return userApi.serviceTerms(tags?.joinToString(","), result)
            .compose(RxOperations.handleApiError())
            .compose(authOperations.handleApiError())
    }

    /**
     * 서비스 약관 동의 철회하기<br>Revoke consent for service terms
     *
     * @param tags 서비스 약관 태그 목록<br>Tags of service terms
     */
    fun revokeServiceTerms(tags: List<String>): Single<UserRevokedServiceTerms> {
        return userApi.revokeServiceTerms(tags.joinToString(","))
            .compose(RxOperations.handleApiError())
            .compose(authOperations.handleApiError())
    }

    /**
     * @suppress
     */
    fun appsShippingAddresses(
        context: Context,
        path: String,
        addressId: Long? = null,
    ): Single<Long> = AuthApiClient.rx.refreshToken()
        .flatMap { AuthApiClient.rx.agt() }
        .flatMap { agt ->
            val uriUtility = UriUtility()
            val continueUrl =
                uriUtility.shippingAddressUrl(path, addressId)
            val url = uriUtility.kpidtUrl(agt, continueUrl.toString())

            Single.create { emitter ->
                val resultReceiver = RxResultReceiver.create(
                    emitter = emitter,
                    identifier = "Apps",
                    parseResponse = { uri -> uri.getQueryParameter(Constants.ADDRESS_ID)?.toLong() },
                    parseError = { uri ->
                        val code =
                            uri.getQueryParameter(com.kakao.sdk.common.Constants.ERROR_CODE)
                        val message =
                            uri.getQueryParameter(com.kakao.sdk.common.Constants.ERROR_MSG)

                        AppsError.create(HttpURLConnection.HTTP_MOVED_TEMP, code, message)
                    },
                    isError = { uri -> uri.getQueryParameter(Constants.APPS_RESULT_STATUS) == Constants.APPS_RESULT_ERROR },
                )

                val intent = IntentFactory.apps<AppsHandlerActivity>(context, resultReceiver, url)
                context.startActivity(intent)
            }
        }

    companion object {
        /**
         * User API 를 호출하기 위한 rx singleton
         */
        @JvmStatic
        val instance by lazy { UserApiClient.rx }
    }
}

/**
 * 간편한 API 요청을 위해 제공되는 싱글톤 객체<br>A singleton object to call APIs easier
 */
val UserApiClient.Companion.rx by lazy { RxUserApiClient() }
