/*
  Copyright 2021 Kakao Corp.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 */
package com.kakao.sdk.auth

import com.kakao.sdk.auth.model.AccessTokenResponse
import com.kakao.sdk.auth.model.AgtResponse
import com.kakao.sdk.auth.model.CertTokenInfo
import com.kakao.sdk.auth.model.CertType
import com.kakao.sdk.auth.model.IdentifyItem
import com.kakao.sdk.auth.model.OAuthToken
import com.kakao.sdk.auth.model.PrepareResponse
import com.kakao.sdk.auth.network.kauth
import com.kakao.sdk.common.KakaoSdk
import com.kakao.sdk.common.model.ApplicationInfo
import com.kakao.sdk.common.model.ApprovalType
import com.kakao.sdk.common.model.AuthError
import com.kakao.sdk.common.model.AuthErrorCause
import com.kakao.sdk.common.model.AuthErrorResponse
import com.kakao.sdk.common.model.ClientError
import com.kakao.sdk.common.model.ClientErrorCause
import com.kakao.sdk.common.model.ContextInfo
import com.kakao.sdk.common.util.KakaoJson
import com.kakao.sdk.network.ApiFactory
import retrofit2.Call
import retrofit2.Callback
import retrofit2.HttpException
import retrofit2.Response

/**
 * @suppress
 */

class AuthApiManager (
    private val authApi: AuthApi = ApiFactory.kauth.create(AuthApi::class.java),
    /** @suppress */ val tokenManagerProvider: TokenManagerProvider = TokenManagerProvider.instance,
    /** @suppress */ val applicationInfo: ApplicationInfo = KakaoSdk.applicationContextInfo,
    /** @suppress */ val contextInfo: ContextInfo = KakaoSdk.applicationContextInfo,
    /** @suppress */ val approvalType: ApprovalType = KakaoSdk.approvalType
) {

    /**
     * @suppress
     */
    internal fun hasToken(): Boolean {
        return tokenManagerProvider.manager.getToken() != null
    }

    /**
     * @suppress
     */
    internal fun issueAccessToken(
        code: String,
        codeVerifier: String? = null,
        callback: (token: OAuthToken?, error: Throwable?) -> Unit
    ) {
        authApi.issueAccessToken(
            clientId = applicationInfo.appKey,
            androidKeyHash = contextInfo.signingKeyHash,
            code = code,
            redirectUri = applicationInfo.redirectUri,
            codeVerifier = codeVerifier,
            approvalType = approvalType.value
        ).enqueue(object : Callback<AccessTokenResponse> {
            override fun onFailure(call: Call<AccessTokenResponse>, t: Throwable) {
                callback(null, t)
            }

            override fun onResponse(
                call: Call<AccessTokenResponse>,
                response: Response<AccessTokenResponse>
            ) {
                if (response.isSuccessful) {
                    response.body()?.let {
                        val token = OAuthToken.fromResponse(it)
                        tokenManagerProvider.manager.setToken(token)
                        callback(token, null)
                        return
                    }
                    callback(null, ClientError(ClientErrorCause.Unknown, "No body"))
                } else {
                    callback(null, translateError(HttpException(response)))
                }
            }
        })
    }

    /**
     * @suppress
     */
    internal fun issueAccessTokenWithCert(
        code: String,
        codeVerifier: String? = null,
        callback: (certTokenInfo: CertTokenInfo?, error: Throwable?) -> Unit
    ) {
        authApi.issueAccessToken(
            clientId = applicationInfo.appKey,
            androidKeyHash = contextInfo.signingKeyHash,
            code = code,
            redirectUri = applicationInfo.redirectUri,
            codeVerifier = codeVerifier,
            approvalType = approvalType.value
        ).enqueue(object : Callback<AccessTokenResponse> {
            override fun onFailure(call: Call<AccessTokenResponse>, t: Throwable) {
                callback(null, t)
            }

            override fun onResponse(
                call: Call<AccessTokenResponse>,
                response: Response<AccessTokenResponse>
            ) {
                if (response.isSuccessful) {
                    response.body()?.let {
                        val token = OAuthToken.fromResponse(it)
                        val txId = it.txId
                        if (txId == null) {
                            callback(null, ClientError(
                                ClientErrorCause.Unknown,
                                "txId is null"
                            ))
                            return
                        }

                        tokenManagerProvider.manager.setToken(token)
                        callback(CertTokenInfo(token, txId),null)
                        return
                    }
                    callback(null, ClientError(ClientErrorCause.Unknown, "No body"))
                } else {
                    callback(null, translateError(HttpException(response)))
                }
            }
        })
    }


    /**
     * @suppress
     */
    @JvmOverloads
    internal fun refreshToken(
        oldToken: OAuthToken = tokenManagerProvider.manager.getToken() ?: throw ClientError(
            ClientErrorCause.TokenNotFound, "Refresh token not found. You must login first."
        ),
        callback: (token: OAuthToken?, error: Throwable?) -> Unit
    ) {
        authApi.refreshToken(
            clientId = applicationInfo.appKey,
            androidKeyHash = contextInfo.signingKeyHash,
            refreshToken = oldToken.refreshToken,
            approvalType = approvalType.value
        ).enqueue(object : Callback<AccessTokenResponse> {
            override fun onFailure(call: Call<AccessTokenResponse>, t: Throwable) {
                callback(null, t)
            }

            override fun onResponse(
                call: Call<AccessTokenResponse>,
                response: Response<AccessTokenResponse>
            ) {
                if (response.isSuccessful) {
                    response.body()?.let {
                        val token = OAuthToken.fromResponse(it, oldToken)
                        tokenManagerProvider.manager.setToken(token)
                        callback(token, null)
                        return
                    }
                    callback(null, ClientError(ClientErrorCause.Unknown, "No body"))
                } else {
                    callback(null, translateError(HttpException(response)))
                }
            }

        })
    }

    /**
     * @suppress
     */
    internal fun agt(
        callback: (agt: String?, error: Throwable?) -> Unit
    ) {
        tokenManagerProvider.manager.getToken()?.accessToken?.let { accessToken ->
            authApi.agt(
                clientId = applicationInfo.appKey,
                accessToken = accessToken
            ).enqueue(object : Callback<AgtResponse> {
                override fun onFailure(call: Call<AgtResponse>, t: Throwable) {
                    callback(null, t)
                }

                override fun onResponse(call: Call<AgtResponse>, response: Response<AgtResponse>) {
                    response.body()?.let { agtResponse ->
                        callback(agtResponse.agt, null)
                        return
                    }
                    callback(null, translateError(HttpException(response)))
                }
            })
        } ?: callback(
            null,
            ClientError(
                ClientErrorCause.TokenNotFound,
                "Access token not found. You must login first."
            )
        )
    }

    /**
     * @suppress
     */
    internal fun refreshToken(
        oldToken: OAuthToken = tokenManagerProvider.manager.getToken() ?: throw ClientError(
            ClientErrorCause.TokenNotFound, "Refresh token not found. You must login first."
        )
    ): OAuthToken {
        val response =
            authApi.refreshToken(
                clientId = applicationInfo.appKey,
                androidKeyHash = contextInfo.signingKeyHash,
                refreshToken = oldToken.refreshToken,
                approvalType = approvalType.value
            ).execute()

        val token = response.body()?.let {
            OAuthToken.fromResponse(it, oldToken)
        } ?: throw translateError(HttpException(response))

        tokenManagerProvider.manager.setToken(token)
        return token
    }

    internal fun prepare(
        settleId: String? = null,
        identifyItems: List<IdentifyItem>? = null,
        signData: String? = null,
        txId: String? = null,
        certType: CertType,
        callback: (kauthTxId: String?, error: Throwable?) -> Unit,
    ) {
        authApi.prepare(
            clientId = applicationInfo.appKey,
            settleId = settleId,
            signIdentifyItems = if (identifyItems.isNullOrEmpty()) null
            else identifyItems.joinToString(",") { it.name.lowercase() },
            signData = signData,
            txId = txId,
            certType = certType.value
        ).enqueue(object : Callback<PrepareResponse> {
            override fun onFailure(call: Call<PrepareResponse>, t: Throwable) {
                callback(null, t)
            }

            override fun onResponse(call: Call<PrepareResponse>, response: Response<PrepareResponse>) {
                if (response.isSuccessful) {
                    response.body()?.let {
                        callback(it.kauthTxId, null)
                        return
                    }
                    callback(null, ClientError(ClientErrorCause.Unknown, "No body"))
                } else {
                    callback(null, translateError(HttpException(response)))
                }
            }
        })
    }

    companion object {
        /**
         * @suppress
         */
        fun translateError(t: Throwable): Throwable {
            try {
                if (t is HttpException) {
                    val errorString = t.response()?.errorBody()?.string()
                    val response =
                        KakaoJson.fromJson<AuthErrorResponse>(
                            errorString!!,
                            AuthErrorResponse::class.java
                        )
                    val cause =
                        kotlin.runCatching {
                            KakaoJson.fromJson<AuthErrorCause>(
                                response.error,
                                AuthErrorCause::class.java
                            )
                        }.getOrDefault(
                            AuthErrorCause.Unknown
                        )
                    return AuthError(t.code(), cause, response)
                }
                return t
            } catch (unexpected: Throwable) {
                return unexpected
            }
        }

        @JvmStatic
        val instance by lazy { AuthApiManager() }
    }
}
