/*
  Copyright 2019 Kakao Corp.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 */
package com.kakao.sdk.auth

import com.kakao.sdk.auth.model.CertTokenInfo
import com.kakao.sdk.auth.model.CertType
import com.kakao.sdk.auth.model.IdentifyItem
import com.kakao.sdk.auth.model.OAuthToken
import com.kakao.sdk.common.model.ClientError
import com.kakao.sdk.common.model.ClientErrorCause

/**
 * 카카오 로그인 인증 및 토큰 관리 클라이언트<br>Client for the authentication and token management through Kakao Login
 */
class AuthApiClient(
    private val manager: AuthApiManager = AuthApiManager.instance,
    /** @suppress */
    val tokenManagerProvider: TokenManagerProvider = TokenManagerProvider.instance,
) {
    /**
     * 토큰 존재 여부 확인하기<br>Check token presence
     */
    fun hasToken(): Boolean = manager.hasToken()

    /**
     * @suppress
     */
    fun issueAccessToken(
        code: String,
        codeVerifier: String? = null,
        callback: (token: OAuthToken?, error: Throwable?) -> Unit,
    ) = manager.issueAccessToken(code, codeVerifier, callback)

    /**
     * @suppress
     * Internal Only
     */
    fun issueAccessTokenWithCert(
        code: String,
        codeVerifier: String? = null,
        callback: (certTokenInfo: CertTokenInfo?, error: Throwable?) -> Unit,
    ) = manager.issueAccessTokenWithCert(code, codeVerifier, callback)

    /**
     * @suppress
     */
    @Deprecated("이 메서드는 더 이상 사용되지 않으므로 refreshToken()을 사용합니다.")
    @JvmOverloads
    fun refreshAccessToken(
        oldToken: OAuthToken = tokenManagerProvider.manager.getToken() ?: throw ClientError(
            ClientErrorCause.TokenNotFound,
            "Refresh token not found. You must login first."
        ),
        callback: (token: OAuthToken?, error: Throwable?) -> Unit,
    ) = manager.refreshToken(oldToken, callback)

    /**
     * 토큰 갱신<br>Refreshes the tokens
     *
     * @param oldToken 토큰 저장소에 저장된 토큰<br>Tokens saved in the token manager
     */
    @JvmOverloads
    fun refreshToken(
        oldToken: OAuthToken = tokenManagerProvider.manager.getToken() ?: throw ClientError(
            ClientErrorCause.TokenNotFound,
            "Refresh token not found. You must login first."
        ),
        callback: (token: OAuthToken?, error: Throwable?) -> Unit,
    ) = manager.refreshToken(oldToken, callback)


    /**
     * @suppress
     */
    fun agt(
        callback: (agt: String?, error: Throwable?) -> Unit,
    ) = manager.agt(callback)

    /**
     * @suppress
     */
    fun prepare(
        settleId: String? = null,
        signData: String? = null,
        txId: String? = null,
        identifyItems: List<IdentifyItem>? = null,
        certType: CertType,
        callback: (kauthTxId: String?, error: Throwable?) -> Unit,
    ) = manager.prepare(settleId, identifyItems, signData, txId, certType, callback)

    companion object {
        @JvmStatic
        val instance by lazy { AuthApiClient() }
    }
}
