/*
  Copyright 2023 Kakao Corp.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 */
package com.kakao.sdk.auth

import android.net.Uri
import com.kakao.sdk.common.util.BaseResultReceiver

/**
 * @suppress
 *
 * 에러만 전달해주는 API일 때 사용
 */
abstract class UnitResultReceiver private constructor(identifier: String) :
    BaseResultReceiver<Unit, (Throwable?) -> Unit>(identifier) {

    override fun onSuccess(response: Unit) {
        this.emitter?.invoke(null)
    }

    override fun onError(error: Throwable) {
        this.emitter?.invoke(error)
    }

    companion object {
        /**
         * @suppress
         */
        fun create(
            emitter: (Throwable?) -> Unit,
            identifier: String,
            parseError: (uri: Uri) -> Throwable,
            isError: (uri: Uri) -> Boolean,
        ): UnitResultReceiver {
            return object : UnitResultReceiver(identifier) {
                override fun parseResponse(url: Uri) = Unit
                override fun parseError(url: Uri): Throwable = parseError(url)
                override fun isError(url: Uri): Boolean = isError(url)
            }.apply {
                setEmitter(emitter)
            }
        }
    }
}

/**
 * @suppress
 *
 * Response, Error 각각 전달하는 API일 때 사용
 */
abstract class SingleResultReceiver<T : Any> private constructor(identifier: String) :
    BaseResultReceiver<T, (T?, Throwable?) -> Unit>(identifier) {

    override fun onSuccess(response: T) {
        this.emitter?.invoke(response, null)
    }

    override fun onError(error: Throwable) {
        this.emitter?.invoke(null, error)
    }

    companion object {
        /**
         * @suppress
         */
        fun <T : Any> create(
            emitter: (T?, Throwable?) -> Unit,
            identifier: String,
            parseResponse: (uri: Uri) -> T?,
            parseError: (uri: Uri) -> Throwable,
            isError: (uri: Uri) -> Boolean,
        ): SingleResultReceiver<T> {
            return object : SingleResultReceiver<T>(identifier) {
                override fun parseResponse(url: Uri): T? = parseResponse(url)
                override fun parseError(url: Uri): Throwable = parseError(url)
                override fun isError(url: Uri): Boolean = isError(url)
            }.apply {
                setEmitter(emitter)
            }
        }
    }
}
