/*
  Copyright 2019 Kakao Corp.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 */
package com.kakao.sdk.auth

import android.app.Activity
import android.content.Intent
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.os.ResultReceiver
import androidx.activity.result.ActivityResult
import androidx.activity.result.ActivityResultCallback
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AppCompatActivity
import com.kakao.sdk.common.model.*
import com.kakao.sdk.common.util.KakaoJson
import com.kakao.sdk.common.util.SdkLog
import com.kakao.sdk.v2.auth.R
import java.net.HttpURLConnection

/**
 * @suppress
 */
class TalkAuthCodeActivity : AppCompatActivity() {
    private lateinit var resultReceiver: ResultReceiver
    private val activityResultLauncher: ActivityResultLauncher<Intent> = registerForActivityResult(
        ActivityResultContracts.StartActivityForResult(),
        activityResultCallback()
    )

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_talk_auth_code)

        try {
            val extras = intent.extras ?: throw IllegalArgumentException("no extras.")
            extras.getBundle(Constants.KEY_BUNDLE)?.run {
                resultReceiver = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                    getParcelable(Constants.KEY_RESULT_RECEIVER, ResultReceiver::class.java)
                } else {
                    @Suppress("DEPRECATION")
                    getParcelable<ResultReceiver>(Constants.KEY_RESULT_RECEIVER) as ResultReceiver
                } ?: throw IllegalStateException()
            }

            val requestCode = extras.getInt(Constants.KEY_REQUEST_CODE)
            SdkLog.i("requestCode: $requestCode")
            val loginIntent = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                extras.getParcelable(Constants.KEY_LOGIN_INTENT, Intent::class.java)
            } else {
                @Suppress("DEPRECATION")
                extras.getParcelable(Constants.KEY_LOGIN_INTENT)
            }
            SdkLog.i("loginIntent:")
            loginIntent?.extras?.run {
                SdkLog.i("\t${Constants.EXTRA_APPLICATION_KEY} : ${getString(Constants.EXTRA_APPLICATION_KEY)}")
                SdkLog.i("\t${Constants.EXTRA_REDIRECT_URI} : ${getString(Constants.EXTRA_REDIRECT_URI)}")
                SdkLog.i("\t${Constants.EXTRA_KA_HEADER} : ${getString(Constants.EXTRA_KA_HEADER)}")
                getBundle(Constants.EXTRA_EXTRAPARAMS)?.run {
                    SdkLog.i("\t${Constants.EXTRA_EXTRAPARAMS}")
                    keySet().map { "\t\t$it : ${getString(it)}" }.forEach { SdkLog.i(it) }
                }
            }
            activityResultLauncher.launch(loginIntent)
        } catch (e: Throwable) {
            SdkLog.e(e)
            sendError(ClientError(ClientErrorCause.Unknown).apply { initCause(e) })
        }
    }

    private fun sendError(exception: KakaoSdkError) {
        if (this::resultReceiver.isInitialized) {
            resultReceiver.send(
                Activity.RESULT_CANCELED,
                Bundle().apply {
                    putSerializable(com.kakao.sdk.common.Constants.KEY_EXCEPTION, exception)
                }
            )
        }
        finish()
    }

    private fun activityResultCallback() = ActivityResultCallback<ActivityResult> { result ->
        val bundle = Bundle()
        if (result.data == null || result.resultCode == Activity.RESULT_CANCELED) {
            sendError(ClientError(ClientErrorCause.Cancelled))
            return@ActivityResultCallback
        }
        if (result.resultCode == Activity.RESULT_OK) {
            val extras = result.data?.extras
            if (extras == null) {
                // no result returned from kakaotalk
                sendError(
                    ClientError(
                        ClientErrorCause.Unknown,
                        "No result from KakaoTalk."
                    )
                )
                return@ActivityResultCallback
            }
            val error = extras.getString(EXTRA_ERROR_TYPE)
            val errorDescription = extras.getString(EXTRA_ERROR_DESCRIPTION)
            if (error == "access_denied") {
                sendError(ClientError(ClientErrorCause.Cancelled))
                return@ActivityResultCallback
            }
            if (error != null) {
                val cause = KakaoJson.fromJson(error, AuthErrorCause::class.java)
                    ?: AuthErrorCause.Unknown
                sendError(
                    AuthError(
                        HttpURLConnection.HTTP_MOVED_TEMP,
                        cause,
                        AuthErrorResponse(
                            error,
                            errorDescription ?: "no error description"
                        )
                    )
                )
                return@ActivityResultCallback
            }
            bundle.putParcelable(
                com.kakao.sdk.common.Constants.KEY_URL,
                Uri.parse(extras.getString(Constants.EXTRA_REDIRECT_URL))
            )
            resultReceiver.send(Activity.RESULT_OK, bundle)
            finish()
            overridePendingTransition(0, 0)
            return@ActivityResultCallback
        }
        throw IllegalArgumentException()
    }

    val EXTRA_ERROR_TYPE = "com.kakao.sdk.talk.error.type"
    val EXTRA_ERROR_DESCRIPTION = "com.kakao.sdk.talk.error.description"

    val NOT_SUPPORT_ERROR = "NotSupportError" // KakaoTalk installed but not signed up
    val UNKNOWN_ERROR = "UnknownError" // No redirect url
    val PROTOCOL_ERROR = "ProtocolError" // Wrong parameters provided
    val APPLICATION_ERROR = "ApplicationError" // Empty redirect url
    val AUTH_CODE_ERROR = "AuthCodeError"
    val CLIENT_INFO_ERROR = "ClientInfoError" // Could not fetch app info

}
