/*
  Copyright 2019 Kakao Corp.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 */
package com.kakao.sdk.user.model

import com.google.gson.annotations.JsonAdapter
import com.google.gson.annotations.SerializedName
import com.kakao.sdk.common.json.KakaoIntDateTypeAdapter
import com.kakao.sdk.common.json.UnknownValue
import java.util.*

/**
 * 배송지 가져오기 응답<br>Response for Retrieve shipping address
 *
 * @property userId 회원번호<br>Service user ID
 * @property needsAgreement 사용자 동의 시 배송지 제공 가능 여부<br>Whether `shippingAddresses` can be provided under user consent
 * @property shippingAddresses 배송지 목록<br>List of shipping addresses
 */
data class UserShippingAddresses(
    val userId: Long?,
    @SerializedName("shipping_addresses_needs_agreement") val needsAgreement: Boolean?,
    val shippingAddresses: List<ShippingAddress>?,
)

/**
 * 배송지 정보<br>Shipping address information
 *
 * @property id 배송지 ID<br>Shipping address ID
 * @property name 배송지 이름<br>Name of shipping address
 * @property isDefault 기본 배송지 여부<br>Whether shipping address is default
 * @property updatedAt 수정 시각<br>Updated time
 * @property type 배송지 타입<br>Shipping address type
 * @property baseAddress 우편번호 검색 시 채워지는 기본 주소<br>Base address that is automatically input when searching for a zipcode
 * @property detailAddress 기본 주소에 추가하는 상세 주소<br>Detailed address that a user adds to the base address
 * @property receiverName 수령인 이름<br>Recipient name
 * @property receiverPhoneNumber1 수령인 연락처<br>Recipient phone number
 * @property receiverPhoneNumber2 수령인 추가 연락처<br>Additional recipient phone number
 * @property zoneNumber 신주소 우편번호<br>New type of 5-digit postal code for a road name address system
 * @property zipCode 구주소 우편번호<br>Old type of 6-digit postal code for an administrative address system
 */
data class ShippingAddress(
    val id: Long,
    val name: String?,
    val isDefault: Boolean,
    @JsonAdapter(KakaoIntDateTypeAdapter::class) val updatedAt: Date?,
    val type: ShippingAddressType?,
    val baseAddress: String?,
    val detailAddress: String?,
    val receiverName: String?,
    val receiverPhoneNumber1: String?,
    val receiverPhoneNumber2: String?,
    val zoneNumber: String?,
    val zipCode: String?,
)

/**
 * 배송지 타입<br>Shipping address type
 */
enum class ShippingAddressType {
    
    /** 구주소<br>Administrative address */
    OLD,

    /** 신주소<br>Road name address */
    NEW,

    /**
     * 알 수 없음<br>Unknown
     */
    @UnknownValue
    UNKNOWN
}
