/*
  Copyright 2021 Kakao Corp.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 */
package com.kakao.sdk.user.model

import android.os.Parcelable
import kotlinx.parcelize.Parcelize

/**
 * 사용자 동의 내역<br>User consent history
 *
 * @property id 회원번호<br>Service user ID
 * @property scopes 앱의 동의항목 목록<br>List of scopes in the app
 */
@Parcelize
data class ScopeInfo(val id: Long, val scopes: List<Scope>?) : Parcelable

/**
 * 동의항목 정보<br>Scope information
 *
 * @property id 동의항목 ID<br>Scope ID
 * @property displayName 사용자 동의 화면에 출력되는 동의항목의 이름 또는 설명<br>Name or description of the scope displayed on the Consent screen.
 * @property type 동의항목 타입<br>Type of the scope
 * @property using 동의항목 사용 여부<br>Whether your app is using the scope
 * @property delegated 카카오가 관리하지 않는 위임 동의항목인지 여부, 현재 사용 중인 동의항목만 응답에 포함<br>Whether the scope is not managed by Kakao, and only the currently used consent is included in the response
 * @property agreed 동의 여부<br>The consent status of the service terms
 * @property revocable 동의 철회 가능 여부<br>Whether the scope can be revoked
 */
@Parcelize
data class Scope(
    val id: String,
    val displayName: String,
    val type: ScopeType,
    val using: Boolean,
    val delegated: Boolean?,
    val agreed: Boolean,
    val revocable: Boolean?,
) : Parcelable

/**
 * 동의항목 타입<br>Scope type
 */
enum class ScopeType {
    /**
     * 개인정보 보호 동의항목<br>Scope for personal information
     */
    PRIVACY,

    /**
     * 접근권한 관리 동의항목<br>Scope for permission
     */
    SERVICE
}
