/*
  Copyright 2019 Kakao Corp.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 */
package com.kakao.sdk.user.model

import android.os.Parcelable
import com.google.gson.annotations.SerializedName
import com.kakao.sdk.common.json.UnknownValue
import kotlinx.parcelize.Parcelize
import java.util.Date

/**
 * 사용자 정보 가져오기 응답<br>Response for Retrieve user information
 *
 * @property id 회원번호<br>Service user ID
 * @property properties 사용자 프로퍼티<br>User properties
 * @property kakaoAccount 카카오계정 정보<br>Kakao Account information
 * @property groupUserToken 그룹에서 맵핑 정보로 사용할 수 있는 값<br>Token to map user in the group apps
 * @property connectedAt 서비스에 연결 완료된 시각, UTC<br>Time connected to the service, UTC
 * @property synchedAt 카카오싱크 간편가입을 통해 로그인한 시각, UTC<br>The time when the user is logged in through Kakao Sync Simple Signup, UTC
 * @property hasSignedUp 연결하기 호출의 완료 여부<br>Whether the user is completely linked with the app
 * @property uuid 다른 사용자의 친구 정보에서 보여지는 해당 사용자의 고유 ID<br>Unique ID for the friend information
 */
@Parcelize
data class User(
    val id: Long?,
    val properties: Map<String, String>?,
    val kakaoAccount: Account?,
    val groupUserToken: String?,
    val connectedAt: Date?,
    val synchedAt: Date?,
    val hasSignedUp: Boolean?,
    val uuid: String?,
) : Parcelable

/**
 * 카카오계정 정보<br>Kakao Account information
 *
 * @property profileNeedsAgreement 사용자 동의 시 프로필 제공 가능 여부<br>Whether `profile` can be provided under user consent
 * @property profileNicknameNeedsAgreement 사용자 동의 시 닉네임 제공 가능 여부<br>Whether `profileNickname` can be provided under user consent
 * @property profileImageNeedsAgreement 사용자 동의 시 프로필 사진 제공 가능 여부<br>Whether `profileImage` can be provided under user consent
 * @property profile 프로필 정보<br>Profile information
 * @property nameNeedsAgreement 사용자 동의 시 이름 제공 가능 여부<br>Whether `name` can be provided under user consent
 * @property name 카카오계정 이름<br>Name of Kakao Account
 * @property emailNeedsAgreement 사용자 동의 시 카카오계정 대표 이메일 제공 가능 여부<br>Whether `email` can be provided under user consent
 * @property isEmailValid 이메일 유효 여부<br>Whether email address is valid
 * @property isEmailVerified 이메일 인증 여부<br>Whether email address is verified
 * @property email 카카오계정 대표 이메일<br>Representative email of Kakao Account
 * @property ageRangeNeedsAgreement 사용자 동의 시 연령대 제공 가능 여부<br>Whether `age` can be provided under user consent
 * @property ageRange 연령대<br>Age range
 * @property birthyearNeedsAgreement 사용자 동의 시 출생 연도 제공 가능 여부<br>Whether `birthyear` can be provided under user consent
 * @property birthyear 출생 연도, YYYY 형식<br>Birthyear in YYYY format
 * @property birthdayNeedsAgreement 사용자 동의 시 생일 제공 가능 여부<br>Whether `birthday` can be provided under user consent
 * @property birthday 생일, MMDD 형식<br>Birthday in MMDD format
 * @property birthdayType 생일 타입<br>Birthday type
 * @property isLeapMonth 생일의 윤달 여부<br>Whether the birthday falls on a leap month
 * @property genderNeedsAgreement 사용자 동의 시 성별 제공 가능 여부<br>Whether `gender` can be provided under user consent
 * @property gender 성별<br>Gender
 * @property legalNameNeedsAgreement 사용자 동의 시 실명 제공 가능 여부<br>Whether `legalName` can be provided under user consent
 * @property legalName 실명<br>Legal name
 * @property legalGenderNeedsAgreement 사용자 동의 시 법정 성별 제공 가능 여부<br>Whether `legalGender` can be provided under user consent
 * @property legalGender 법정 성별<br>Legal gender
 * @property legalBirthDateNeedsAgreement 사용자 동의 시 법정 생년월일 제공 가능 여부<br>Whether `isKorean` can be provided under user consent
 * @property legalBirthDate 법정 생년월일, yyyyMMDD 형식<br>Legal birth date in yyyyMMDD format
 * @property phoneNumberNeedsAgreement 사용자 동의 시 전화번호 제공 가능 여부<br>Whether `phoneNumber` can be provided under user consent
 * @property phoneNumber 카카오계정의 전화번호<br>Phone number of Kakao Account
 * @property isKoreanNeedsAgreement 사용자 동의 시 내외국인 제공 가능 여부<br>Whther consent to `isKorean` can be provided under user consent
 * @property isKorean 본인인증을 거친 내국인 여부<br>Whether the user is Korean
 */
@Parcelize
data class Account(
    val profileNeedsAgreement: Boolean?,
    val profileNicknameNeedsAgreement: Boolean?,
    val profileImageNeedsAgreement: Boolean?,
    val profile: Profile?,

    val nameNeedsAgreement: Boolean?,
    val name: String?,

    val emailNeedsAgreement: Boolean?,
    val isEmailValid: Boolean?,
    val isEmailVerified: Boolean?,
    val email: String?,

    val ageRangeNeedsAgreement: Boolean?,
    val ageRange: AgeRange?,

    val birthyearNeedsAgreement: Boolean?,
    val birthyear: String?,

    val birthdayNeedsAgreement: Boolean?,
    val birthday: String?,
    val birthdayType: BirthdayType?,
    val isLeapMonth: Boolean?,

    val genderNeedsAgreement: Boolean?,
    val gender: Gender?,

    val legalNameNeedsAgreement: Boolean?,
    val legalName: String?,

    val legalBirthDateNeedsAgreement: Boolean?,
    val legalBirthDate: String?,

    val legalGenderNeedsAgreement: Boolean?,
    val legalGender: Gender?,

    val phoneNumberNeedsAgreement: Boolean?,
    val phoneNumber: String?,

    val isKoreanNeedsAgreement: Boolean?,
    val isKorean: Boolean?,
) : Parcelable


/**
 * 프로필 정보<br>Profile information
 *
 * @property nickname 닉네임<br>Nickname
 * @property profileImageUrl 프로필 사진 URL<br>Profile image URL
 * @property thumbnailImageUrl 프로필 미리보기 이미지 URL<br>Thumbnail image URL
 * @property isDefaultImage 프로필 사진 URL이 기본 프로필 사진 URL인지 여부<br>Whether the default image is used for profile image
 */
@Parcelize
data class Profile(
    val nickname: String?,
    val profileImageUrl: String?,
    val thumbnailImageUrl: String?,
    val isDefaultImage: Boolean?,
) : Parcelable

/**
 * 연령대<br>Age range
 */
enum class AgeRange {
    /**
     * 0세~9세<br>0 to 9 years old
     */
    @SerializedName("0~9")
    AGE_0_9,

    /**
     * 10세~14세<br>10 to 14 years old
     */
    @SerializedName("10~14")
    AGE_10_14,

    /**
     * 15세~19세<br>15 to 19 years old
     */
    @SerializedName("15~19")
    AGE_15_19,

    /**
     * 20세~29세<br>20 to 29 years old
     */
    @SerializedName("20~29")
    AGE_20_29,

    /**
     * 30세~39세<br>30 to 39 years old
     */
    @SerializedName("30~39")
    AGE_30_39,

    /**
     * 40세~49세<br>40 to 49 years old
     */
    @SerializedName("40~49")
    AGE_40_49,

    /**
     * 50세~59세<br>50 to 59 years old
     */
    @SerializedName("50~59")
    AGE_50_59,

    /**
     * 60세~69세<br>60 to 69 years old
     */
    @SerializedName("60~69")
    AGE_60_69,

    /**
     * 70세~79세<br>70 to 79 years old
     */
    @SerializedName("70~79")
    AGE_70_79,

    /**
     * 80세~89세<br>80 to 89 years old
     */
    @SerializedName("80~89")
    AGE_80_89,

    /**
     * 90세 이상<br>Over 90 years old
     */
    @SerializedName("90~")
    AGE_90_ABOVE,

    /**
     * 알 수 없음<br>Unknown
     */
    @UnknownValue
    UNKNOWN;
}

/**
 * 성별<br>Gender
 */
enum class Gender {
    /**
     * 여자<br>Female
     */
    @SerializedName("female")
    FEMALE,

    /**
     * 남자<br>Male
     */
    @SerializedName("male")
    MALE,

    /**
     * 알 수 없음<br>Unknown
     */
    @UnknownValue
    UNKNOWN
}

/**
 * 생일 타입<br>Birthday type
 */
enum class BirthdayType {
    /**
     * 양력<br>Solar
     */
    @SerializedName("SOLAR")
    SOLAR,

    /**
     * 음력<br>Lunar
     */
    @SerializedName("LUNAR")
    LUNAR,

    /**
     * 알 수 없음<br>Unknown
     */
    @UnknownValue
    UNKNOWN
}
